import asyncio

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar

from predefined_tasks.common.helper import send_to_gc

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from urpc_enum.measurementparameter import MeasurementParameter

from bodhi.common.firmware_util import *

meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit


async def pmt1_measurement(dl=0.3, hv=0.5, window_ms=1000, iterations=10, delay=1.0):
    
    GlobalVar.set_stop_gc(False)
    await send_to_gc(f"Starting PMT1 measurement")

    delay = delay - window_ms / 1000
    
    try:
        await send_to_gc(f"Starting Firmware")
        await start_firmware('eef')
        await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl)
        await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv)
        await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 1)
        await asyncio.sleep(1.0)
    
        op_id = 'pmt1_measurement'
        meas_unit.ClearOperations()
        await load_pmt1_measurement(op_id, window_ms)

        for i in range(iterations):

            if GlobalVar.get_stop_gc():
                return f"pmt1_measurement stopped by user"
            
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            pmt1_cnt = results[0] + (results[1] << 32)
            pmt1_al  = results[4]
            pmt1_ah  = results[5]

            await send_to_gc(f"counting: {pmt1_cnt:9.0f} ; analog_low: {pmt1_al:5.0f} ; analog_high: {pmt1_ah:5.0f}", log=True)

            if delay > 0:
                await asyncio.sleep(delay)

        return f"pmt1_measurement done"

    finally:
        await meas_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0)


async def load_pmt1_measurement(op_id, window_ms=1.0):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    fixed_range = 2000          #  20 us
    reset_switch_delay = 2000   #  20 us
    input_gate_delay = 100      #   1 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah

    seq_gen.SetSignals(OutputSignal.HVGatePMT1)

    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_autorange)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_with_fixed_range)
    
    seq_gen.TimerWait()
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 6)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

